﻿using NVCC.Models;
using NVCC.WebUI.Infrastructure;
using NVCC.WebUI.Models;
using System.Diagnostics;
using System.Linq;
using System.Text.RegularExpressions;
using System.Web.Mvc;

namespace NVCC.WebUI.Controllers
{
    public class HomeController : Controller
    {
        private readonly IUserService _userService;
        private readonly IPatientProfileService _patientProfileService;

        public HomeController(IPatientProfileService patientProfileService,
                              IUserService userService)
        {
            _userService = userService;
            _patientProfileService = patientProfileService;
        }

        [HttpGet]
        [NvccAuthorize]
        public ActionResult Index()
        {
            User user = _userService.GetUser();
            if (user.ActiveFacilities().Count() == 0)
            {
                // ought to have already been caught by the NvccAuthorize attribute, but don't assume
                return RedirectToAction("Login", "User");
            }
            return View(new HomePageViewModel { ActiveFacilities = user.ActiveFacilities(), sta3n = user.CurrentDefaultFacility });
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        [NvccAuthorize]
        public ActionResult Index([Bind(Include = "PatientSSN, sta3n, IncludeVistAImaging")] HomePageViewModel viewModel)
        {
            User user = _userService.GetUser();
            // populate rest of view model so that it can be passed to a View if need be
            viewModel.ActiveFacilities = user.ActiveFacilities();

            var modelState = CheckSsn(viewModel);
            if (modelState.IsValid == false)
                return View(viewModel);

            if (user.ActiveFacilities().Where(f => (f.sta3n == viewModel.sta3n)).Count() == 0)
            {
                ModelState.AddModelError("sta3n", "Station selected is not active");
                return View(viewModel);
            }

            var stopwatch = new Stopwatch();
            stopwatch.Start();
            var patient = _patientProfileService.GetPatient(viewModel.PatientSSN, viewModel.sta3n);
            stopwatch.Stop();
            if (patient == null)
            {
                ModelState.AddModelError("PatientSSN", "Patient not found for SSN given.");
                return View(viewModel);
            }

            _patientProfileService.LogItem(stopwatch, patient, user.DomainPlusNetworkUserName, "lookup");

            if (HttpContextManager.Current.Session != null)
            {
                HttpContextManager.Current.Session[patient.PatientSid.ToString()] = patient;
            }

            user.CurrentDefaultFacility = viewModel.sta3n;

            return RedirectToAction("Index", "PatientProfile", new { patientSid = patient.PatientSid, includeVistAImaging = viewModel.IncludeVistAImaging });
        }

        private ModelStateDictionary CheckSsn(HomePageViewModel viewModel)
        {
            if (viewModel == null)
            {
                ModelState.AddModelError("viewModel", "Nothing to see");
                return ModelState;
            }
            if (string.IsNullOrEmpty(viewModel.PatientSSN))
            {
                ModelState.AddModelError("PatientSSN", "SSN field cannot be blank");
                return ModelState;
            }
            viewModel.PatientSSN = viewModel.PatientSSN.Replace("-", "").Replace(" ", "").Trim();
            var r = new Regex("^\\d*$");
            if (!r.IsMatch(viewModel.PatientSSN))
            {
                ModelState.AddModelError("PatientSSN", "SSN must be only numbers, spaces, and dashes");
            }
            else if (viewModel.PatientSSN.Length != 9)
            {
                ModelState.AddModelError("PatientSSN", "SSN must have exactly 9 digits");

            }
            return ModelState;
        }

        [AllowAnonymous]
        public ActionResult Unauthorized()
        {
            return View();
        }

        //Application level error
        [AllowAnonymous]
        public ActionResult Error()
        {
            ViewBag.Message = TempData["Message"] as string;
            return View();
        }

    }
}